%KM-FILM model for recreating Figures 2a-b, 3a-f and 4 in Lakey et al. Indoor
%Air, 31 (6), 2070-2083 (2021)

%Layers
%1: mfp, Mean free path next to the surface
%2: L_BL_close+1, Boundary layer near the surface
%L_BL_close+2:L_BL_close+L_BL_far+1, Boundary layer further from the surface
%L_BL_close+L_BL_far+2, Gas phase
%L_BL_close+L_BL_far+3, Molecules adsorbed to the surface
%L_BL_close+L_BL_far+4:L_BL_close+L_BL_far+adsorption_layers+3, Molecules adsorbed to other molecules

clear all
warning off all
%Species: 1.DEHP, 2.DEHT, 3.DINCH, 4.DINP
global W sigma adsorption_layers L_BL_close L_BL_far JJ A_surf
global a0_room_surf kd_room_surf a0 kd V_BL_close
global V_BL_far delta_BL_close delta_BL_far Dg
global V_w NCom mfp delta_x A ka 

for JJ=1:17 %loop for each of the 20 experiments 
    %Note that the order of experiments are different to Lakey et al. (2021)
    JJoutput=JJ %Output to monitor the progress of the model

%%%%General parameters%%%%
NCom=7; %Number of species
NA=6.02e23; %Avogadro's number (mol-1)
R=8.31e7; %Gas constant (g cm2 s-2 K-1 mol-1)
T=298; %Temperature (K)

%%%%Parameters for different species%%%%
%DEHP
Mx(1)=390.6;  % Molar mass of DEHP (g mol-1)
Dg(1)=0.0264;% Gas-phase diffusion coefficient from SPARC calculator (cm2 s-1)
a0_room_surf(1)=1; %Surface mass accommodation on an adsorbate free surface of the test material
a0(1)=1; %Surface mass accommodation to adsorbed organics

%DEHT
Mx(2)=390.6; %Molar mass of DEHT (g mol-1)
Dg(2)=0.0264;% Gas-phase diffusion coefficient from SPARC calculator (cm2 s-1)
a0_room_surf(2)=1;%Surface mass accommodation on an adsorbate free surface of the test material
a0(2)=1;%Surface mass accommodation to adsorbed organics

%DINCH
Mx(3)=424.7; %Molar mass of DINCH (g mol-1)
Dg(3)=0.0238;% Gas-phase diffusion coefficient from SPARC calculator (cm2 s-1)
a0_room_surf(3)=1;%Surface mass accommodation on an adsorbate free surface of the test material
a0(3)=1;%Surface mass accommodation to adsorbed organics

%DINP
Mx(4)=418.6; %Molar mass of DINP (g mol-1)
Dg(4)=0.0242;% Gas-phase diffusion coefficient from SPARC calculator (cm2 s-1)
a0_room_surf(4)=1;%Surface mass accommodation on an adsorbate free surface of the test material
a0(4)=1;%Surface mass accommodation to adsorbed organics

%DIDP
Mx(5)=446.7; %Molar mass of DIDP (g mol-1)
Dg(5)=0.0227;% Gas-phase diffusion coefficient from SPARC calculator (cm2 s-1)
a0_room_surf(5)=1;%Surface mass accommodation on an adsorbate free surface of the test material
a0(5)=1;%Surface mass accommodation to adsorbed organics

%DEHA
Mx(6)=370.6; %Molar mass of DEHA (g mol-1)
Dg(6)=0.0268;% Gas-phase diffusion coefficient from SPARC calculator (cm2 s-1)
a0_room_surf(6)=1;%Surface mass accommodation on an adsorbate free surface of the test material
a0(6)=1;%Surface mass accommodation to adsorbed organics

%ATBC
Mx(7)=402.5; %Molar mass of ATBC (g mol-1)
Dg(7)=0.0275;% Gas-phase diffusion coefficient from SPARC calculator (cm2 s-1)
a0_room_surf(7)=1;%Surface mass accommodation on an adsorbate free surface of the test material
a0(7)=1;%Surface mass accommodation to adsorbed organics

%Additional parameters
for i=1:NCom
Dens(i)=1; % Density of bulk species (g cm-3)
V_w(i)=Mx(i)/NA/Dens(i); % Volume of one O3 molecule (cm3)
delta_x(i)=V_w(i)^(1/3); % Geometrical diameter of one molecule (cm)
sigma(i)=delta_x(i)^2; % Effective molecular cross section of one molecule (cm2)
W(i)=sqrt(8*R*T/pi/Mx(i)); %Mean thermal velocity of one molecule (cm s-1)
mfp(i)= ((3*Dg(i))/W(i)); %Mean free path (cm)
ka(i)=a0(i)*W(i)/4; %Adsorption rate coefficient (cm s-1)
end

%%%%Parameters for different test materials%%%%
%Roughness factor (Measured values, Note: True surface area= Roughness factor * Area footprint)
if JJ==1
    Roughness_factor=1.028187377; %Aluminum
elseif JJ==2
    Roughness_factor=1.026385676; %Smooth glass
elseif JJ==3
    Roughness_factor=1.79500771; %Rough glass
elseif JJ==4
    Roughness_factor=1.043424195; %Stainless steel
elseif JJ==5
    Roughness_factor=1.12099637; %Acrylic plastic
elseif JJ==13 || JJ==14 || JJ==15 || JJ==16 || JJ==17
    Roughness_factor=2.132340658; %Steel
else
    Roughness_factor=1.234826094; %Stainless steel #2
end
A_surf=1*Roughness_factor; %True surface area (cm2)

if JJ<=10
    Disk_surf_area=3.204738666; %Total surface area of the disk (cm2)
else
    Disk_surf_area=3.879479137; %Total surface area of the disk (cm2)
end

%%%%%%%%%Experimental concentrations%%%%%%%%%
%Gas-phase concentrations next to the source (cm-3)
Source_conc_DEHP=0; %conc. initially set to zero but changed below
Source_conc_DEHT=0;
Source_conc_DINCH=0;
Source_conc_DINP=0;
Source_conc_DIDP=0;
Source_conc_DEHA=0;
Source_conc_ATBC=0;
if JJ<=7
    Source_conc_DEHP=2.5*1e-6/Mx(1)*1e-6*6.02e23; %molec. cm-3
elseif JJ==8
    Source_conc_DEHT=0.56*1e-6/Mx(2)*1e-6*6.02e23;%molec. cm-3
elseif JJ==9
    Source_conc_DIDP=0.08*1e-6/Mx(5)*1e-6*6.02e23;%molec. cm-3
elseif JJ==10
    Source_conc_DEHP=0.02*1e-6/Mx(1)*1e-6*6.02e23;%molec. cm-3
    Source_conc_DEHA=2.2*1e-6/Mx(6)*1e-6*6.02e23;%molec. cm-3
elseif JJ==11
    Source_conc_DEHT=0.8*1e-6/Mx(2)*1e-6*6.02e23;%molec. cm-3
    Source_conc_DINP=0.11*1e-6/Mx(4)*1e-6*6.02e23;%molec. cm-3
elseif JJ==12
    Source_conc_DEHT=0.2*1e-6/Mx(2)*1e-6*6.02e23;%molec. cm-3
    Source_conc_DEHA=1.3*1e-6/Mx(6)*1e-6*6.02e23;%molec. cm-3  
elseif JJ==13
    Source_conc_DEHP=0.25*1e-6/Mx(1)*1e-6*6.02e23;%molec. cm-3
    Source_conc_DEHT=0.21*1e-6/Mx(2)*1e-6*6.02e23;%molec. cm-3
    Source_conc_DIDP=0*1e-6/Mx(5)*1e-6*6.02e23;%molec. cm-3
    Source_conc_DINP=0.06*1e-6/Mx(4)*1e-6*6.02e23;%molec. cm-3
    Source_conc_DINCH=0.011*1e-6/Mx(3)*1e-6*6.02e23;%molec. cm-3
elseif JJ==14
    Source_conc_DEHT=0.4*1e-6/Mx(2)*1e-6*6.02e23;%molec. cm-3
    Source_conc_ATBC=0.15*1e-6/Mx(7)*1e-6*6.02e23;%molec. cm-3
elseif JJ==15
    Source_conc_DEHT=0.42*1e-6/Mx(2)*1e-6*6.02e23;%molec. cm-3
    Source_conc_ATBC=0.95*1e-6/Mx(7)*1e-6*6.02e23;%molec. cm-3
elseif JJ==16
    Source_conc_DEHT=0.4*1e-6/Mx(2)*1e-6*6.02e23;%molec. cm-3
elseif JJ==17
    Source_conc_DEHT=0.29*1e-6/Mx(2)*1e-6*6.02e23;%molec. cm-3
end

%Desorption rate coefficients from surfaces (s-1) (kept consistent between experiments)
kd_room_surf=zeros(NCom,1); kd=zeros(NCom,1);

if JJ==1
kd_room_surf(1)=0.3529; %DEHP desorption from aluminum
elseif JJ==2 
kd_room_surf(1)=0.1765; %DEHP desorption from smooth glass
elseif JJ==3
kd_room_surf(1)=0.0118; %DEHP desorption from rough glass     
elseif JJ==4 || JJ==7 || JJ==10
kd_room_surf(1)=0.0392; %DEHP desorption from stainless steel
elseif JJ==5
kd_room_surf(1)=0.4706; %DEHP desorption from acrylic plastic
elseif JJ==6 || JJ==13
kd_room_surf(1)=0.0118; %DEHP desorption from steel
end
if JJ==1 || JJ==2 || JJ==3 || JJ==4 || JJ==5 || JJ==6 || JJ==7
kd(1)=0.0905;   %DEHP desorption from adsorbed DEHP molecules  
end
if JJ==8 || JJ==11 || JJ==12
kd_room_surf(2)=0.0804;   %DEHT desorption from Stainless steel   
elseif JJ==13 || JJ==14 || JJ==15 || JJ==16 || JJ==17
kd_room_surf(2)=0.0103;  %DEHT desorption from steel     
end
if JJ==8 || JJ==16 || JJ==17
kd(2)=0.0071; %DEHT desorption from adsorbed DEHT molecules       
end
if JJ==14 || JJ==15
kd_room_surf(7)=0.0370; %ATBC from steel     
end
if JJ==10 || JJ==12
kd_room_surf(6)=0.0351; %DEHA from Stainless steel     
end
if JJ==9
kd(5)=0.0024; %DIDP from stainless steel  
kd_room_surf(5)=0.0024; %DIDP desorption from adsorbed DIDP molecules 
end
if JJ==10
kd(1)=0.0352; %DEHP from adsorbed molecules    
kd(6)=0.0622; %DEHA from adsorbed molecules 
end
if JJ==11
kd_room_surf(4)=0.0046; %DINP from stainless steel
kd(2)=0.0780; %DEHT from adsorbed molecules
kd(4)=0.0139; %DINP from adsorbed molecules
end
if JJ==12
kd(2)=0.0067; %DEHT from adsorbed molecules
kd(6)=0.0937; %DEHA from adsorbed molecules
end
if JJ==13
kd_room_surf(4)=0.0157; %DINP from steel
kd_room_surf(3)=0.0168; %DINCH from steel
kd(2)=0.0125; %DEHT from adsorbed molecules
kd(1)=0.0334; %DEHP from adsorbed molecules
kd(3)=0.0168; %DINCH from adsorbed molecules
kd(4)=0.0079; %DINP from adsorbed molecules
end
if JJ==14
kd(2)=0.0148;%DEHT from adsorbed molecules
kd(7)=0.2660;%ATBC from adsorbed molecules
end
if JJ==15
kd(2)=0.0175; %DEHT from adsorbed molecules
kd(7)=0.2660; %ATBC from adsorbed molecules    
end

%Other parameters
adsorption_layers=20; %(note that this does not include the adsorption layer which forms on the original surface)

%%%%%%Boundary layer parameters%%%%%%
%%%%boundary layer thickness%%%%
if JJ<=7
    BL_close=1e-4; %Thickness of the boundary layer close to the surface (cm)
    BL_far=0.14-BL_close; %Thickness of the remaining boundary layer (experimental value, cm)
else
    BL_close=1e-4; %Thickness of the boundary layer close to the surface (cm)
    BL_far=0.0254-BL_close; %Thickness of the remaining boundary layer (experimental value, cm)
end

A=1;%Unit surface area of the boundary layer (cm2)
L_BL_close=5; %Number of layers in the close boundary layer
L_BL_far=5; %Number of layers in the remaining boundary layer
delta_BL_close=BL_close./L_BL_close;%Thickness of 1 layer in the close boundary layer (cm)
delta_BL_far=BL_far./L_BL_far;%Thickness of 1 layer in the remaining boundary layer (cm)
V_BL_close=delta_BL_close.*A;%Volume of 1 layer in the close boundary layer (cm3)
V_BL_far=delta_BL_far.*A;%Volume of 1 layer in the remaining boundary layer (cm3)

% Maximum experimental times for modeling (h)
if JJ<=2
    maxtime=340;
elseif JJ==3
    maxtime=290;
elseif JJ==4
    maxtime=260;
elseif JJ==6
    maxtime=240;
elseif JJ==7 || JJ==9
    maxtime=170;
elseif JJ==8 || JJ==10
    maxtime=180;
elseif JJ==11 || JJ==12
    maxtime=200;
elseif JJ==13
    maxtime=630;
elseif JJ>=16
    maxtime=300;
else
    maxtime=320;
end

%Initial conditions

%%%%%Initial concentrations%%%%%
clear y0
%Initial gas-phase concentrations of the different species (molec. cm-3)
y0=zeros((L_BL_close+L_BL_far+adsorption_layers+3)*NCom,1);
y0((L_BL_close+L_BL_far+adsorption_layers+3)*(1-1)+L_BL_close+L_BL_far+2,1)=Source_conc_DEHP;
y0((L_BL_close+L_BL_far+adsorption_layers+3)*(2-1)+L_BL_close+L_BL_far+2,1)=Source_conc_DEHT;
y0((L_BL_close+L_BL_far+adsorption_layers+3)*(3-1)+L_BL_close+L_BL_far+2,1)=Source_conc_DINCH;
y0((L_BL_close+L_BL_far+adsorption_layers+3)*(4-1)+L_BL_close+L_BL_far+2,1)=Source_conc_DINP;
y0((L_BL_close+L_BL_far+adsorption_layers+3)*(5-1)+L_BL_close+L_BL_far+2,1)=Source_conc_DIDP;
y0((L_BL_close+L_BL_far+adsorption_layers+3)*(6-1)+L_BL_close+L_BL_far+2,1)=Source_conc_DEHA;
y0((L_BL_close+L_BL_far+adsorption_layers+3)*(7-1)+L_BL_close+L_BL_far+2,1)=Source_conc_ATBC;

%***********Solve differential equations*********
clear Y
n=2500; %outputted time steps
t=logspace(-3,log10(3600*maxtime),n);t=t'; %time being investigated (s)
AbsTol=1;
RelTol=1e-3;
options=odeset('AbsTol',AbsTol,'RelTol',RelTol);
[X,Y]=ode23tb(@KM_FILM_adsorption_F,t,y0); %solver

%%%%Outputs%%%%%
t_hour_1(:,JJ)=t(:,1)/3600; %Outputted time (h)
        for i=1:n
            for j=1:adsorption_layers+1
                for m=L_BL_close+L_BL_far+adsorption_layers+3
%Number of molecules in each layer for each time step for a 1 cm2 area footprint
        DEHP_molec_output1(i,j)=(Y(i,(m*(1-1)+L_BL_close+L_BL_far+2+j)));
        DEHT_molec_output1(i,j)=(Y(i,(m*(2-1)+L_BL_close+L_BL_far+2+j)));
        DINCH_molec_output1(i,j)=(Y(i,(m*(3-1)+L_BL_close+L_BL_far+2+j)));
        DINP_molec_output1(i,j)=(Y(i,(m*(4-1)+L_BL_close+L_BL_far+2+j)));
        DIDP_molec_output1(i,j)=(Y(i,(m*(5-1)+L_BL_close+L_BL_far+2+j)));
        DEHA_molec_output1(i,j)=(Y(i,(m*(6-1)+L_BL_close+L_BL_far+2+j)));
        ATBC_molec_output1(i,j)=(Y(i,(m*(7-1)+L_BL_close+L_BL_far+2+j)));
                end
            end
        end

 %Total number of adsorbed molecules for each time step for a 1 cm2 area footprint    
 for i=1:n
 aa_DEHP_molec_output(i,JJ)=sum(DEHP_molec_output1(i,1:adsorption_layers+1));
 aa_DEHT_molec_output(i,JJ)=sum(DEHT_molec_output1(i,1:adsorption_layers+1));
 aa_DINCH_molec_output(i,JJ)=sum(DINCH_molec_output1(i,1:adsorption_layers+1));
 aa_DINP_molec_output(i,JJ)=sum(DINP_molec_output1(i,1:adsorption_layers+1));
 aa_DIDP_molec_output(i,JJ)=sum(DIDP_molec_output1(i,1:adsorption_layers+1));
 aa_DEHA_molec_output(i,JJ)=sum(DEHA_molec_output1(i,1:adsorption_layers+1));
 aa_ATBC_molec_output(i,JJ)=sum(ATBC_molec_output1(i,1:adsorption_layers+1));
 
 %Total nanograms of adsorbed molecules for the total disk area
 aa_DEHP_ng_output(i,JJ)=sum(DEHP_molec_output1(i,1:adsorption_layers+1))*Mx(1)*1e9/6.02e23*Disk_surf_area;
 aa_DEHT_ng_output(i,JJ)=sum(DEHT_molec_output1(i,1:adsorption_layers+1))*Mx(2)*1e9/6.02e23*Disk_surf_area;
 aa_DINCH_ng_output(i,JJ)=sum(DINCH_molec_output1(i,1:adsorption_layers+1))*Mx(3)*1e9/6.02e23*Disk_surf_area;
 aa_DINP_ng_output(i,JJ)=sum(DINP_molec_output1(i,1:adsorption_layers+1))*Mx(4)*1e9/6.02e23*Disk_surf_area;
 aa_DIDP_ng_output(i,JJ)=sum(DIDP_molec_output1(i,1:adsorption_layers+1))*Mx(5)*1e9/6.02e23*Disk_surf_area;
 aa_DEHA_ng_output(i,JJ)=sum(DEHA_molec_output1(i,1:adsorption_layers+1))*Mx(6)*1e9/6.02e23*Disk_surf_area;
 aa_ATBC_ng_output(i,JJ)=sum(ATBC_molec_output1(i,1:adsorption_layers+1))*Mx(7)*1e9/6.02e23*Disk_surf_area;
 end
end

%%%%%%%Experimental measurement data%%%%%%
%Measurements are from Wu et al. Environ. Sci. Tech. 2017, 51 (5),
%2907-2913 and Eichler et al. Environ. Sci. Tech. 2018, 52 (5), 2918-2925.

Data_1=xlsread('Data for MOCCIE', 'All mean concentrations Ks');
Data_2=xlsread('Data for MOCCIE', 'Measurements C0-y0');

%DEHP Aliminum
Time_DEHP_Aluminum=Data_1(1:11,1); %h
Data_DEHP_Aluminum=Data_1(1:11,2); %ng
Error_DEHP_Aluminum=Data_1(1:11,3); %ng

%DEHP Smooth glass
Time_DEHP_Smoothglass=Data_1(15:23,1); %h
Data_DEHP_Smoothglass=Data_1(15:23,2); %ng
Error_DEHP_Smoothglass=Data_1(15:23,3); %ng

%DEHP Rough glass
Time_DEHP_Roughglass=Data_1(27:34,1); %h
Data_DEHP_Roughglass=Data_1(27:34,2); %ng
Error_DEHP_Roughglass=Data_1(27:34,3); %ng

%DEHP Stainless steel #1
Time_DEHP_Stainlesssteel_1=Data_1(38:47,1); %h
Data_DEHP_Stainlesssteel_1=Data_1(38:47,2); %ng
Error_DEHP_Stainlesssteel_1=Data_1(38:47,3); %ng

%DEHP Acrylic plastic
Time_DEHP_Acrylicplastic=Data_1(51:56,1); %h 
Data_DEHP_Acrylicplastic=Data_1(51:56,2); %ng
Error_DEHP_Acrylicplastic=Data_1(51:56,3); %ng

%DEHP on steel
Time_DEHP_steel=Data_1(60:68,1); %h
Data_DEHP_steel=Data_1(60:68,2); %ng
Error_DEHP_steel=Data_1(60:68,3); %ng

%DEHP on stainless steel #2
Time_DEHP_Stainlesssteel_2=Data_1(72:78,1); %h
Data_DEHP_Stainlesssteel_2=Data_1(72:78,2); %ng
Error_DEHP_Stainlesssteel_2=Data_1(72:78,3); %ng

%DEHT from Backpack R0b on stainless steel #2
Time_Backpack_R0b_DEHT=Data_2(1:5,1); %h
Data_Backpack_R0b_DEHT=Data_2(1:5,2); %ng
Error_Backpack_R0b_DEHT=Data_2(1:5,3); %ng

%DIDP from backpack STb on stainless steel #2
Time_Backpack_STb_DIDP=Data_2(8:13,1); %h
Data_Backpack_STb_DIDP=Data_2(8:13,2); %ng
Error_Backpack_STb_DIDP=Data_2(8:13,3); %ng

%DEHP from backpack TAb on stainless steel #2
Time_Backpack_Tab_DEHP=Data_2(16:21,1); %h
Data_Backpack_Tab_DEHP=Data_2(16:21,2); %ng
Error_Backpack_Tab_DEHP=Data_2(16:21,3); %ng

%DEHA from backpack Tab on stainless steel #2
Time_Backpack_Tab_DEHA=Data_2(16:21,1); %h
Data_Backpack_Tab_DEHA=Data_2(16:21,4); %ng
Error_Backpack_Tab_DEHA=Data_2(16:21,5); %ng

%DEHT from backpack TMb on stainless steel #2
Time_Backpack_TMb_DEHT=Data_2(24:30,1); %h
Data_Backpack_TMb_DEHT=Data_2(24:30,2); %ng
Error_Backpack_TMb_DEHT=Data_2(24:30,3); %ng

%DINP from backpack TMb on stainless steel #2
Time_Backpack_TMb_DINP=Data_2(24:30,1); %h
Data_Backpack_TMb_DINP=Data_2(24:30,4); %ng
Error_Backpack_TMb_DINP=Data_2(24:30,5); %ng

%DEHT from backpack WAb1 on stainless steel #2
Time_Backpack_WAb1_DEHT=Data_2(33:39,1); %h
Data_Backpack_WAb1_DEHT=Data_2(33:39,2); %ng
Error_Backpack_WAb1_DEHT=Data_2(33:39,3); %ng

%DEHA from backpack WAb1 on stainless steel #2
Time_Backpack_WAb1_DEHA=Data_2(33:39,1); %h
Data_Backpack_WAb1_DEHA=Data_2(33:39,4); %ng
Error_Backpack_WAb1_DEHA=Data_2(33:39,5); %ng

%DEHP from backpack WAb2 on steel
Time_Backpack_WAb2_DEHP=Data_2(42:47,1); %h
Data_Backpack_WAb2_DEHP=Data_2(42:47,2); %ng
Error_Backpack_WAb2_DEHP=Data_2(42:47,3); %ng

%DEHT from backpack WAb2 on steel
Time_Backpack_WAb2_DEHT=Data_2(42:47,1); %h
Data_Backpack_WAb2_DEHT=Data_2(42:47,4); %ng
Error_Backpack_WAb2_DEHT=Data_2(42:47,5); %ng

%DIDP from backpack WAb2 on steel
Time_Backpack_WAb2_DIDP=Data_2(42:47,1); %h
Data_Backpack_WAb2_DIDP=Data_2(42:47,6); %ng
Error_Backpack_WAb2_DIDP=Data_2(42:47,7); %ng

%DINP from backpack WAb2 on steel
Time_Backpack_WAb2_DINP=Data_2(42:47,1); %h
Data_Backpack_WAb2_DINP=Data_2(42:47,8); %ng
Error_Backpack_WAb2_DINP=Data_2(42:47,9); %ng

%DINCH from backpack WAb2 on steel
Time_Backpack_WAb2_DINCH=Data_2(42:47,1); %h
Data_Backpack_WAb2_DINCH=Data_2(42:47,10); %ng
Error_Backpack_WAb2_DINCH=Data_2(42:47,11); %ng

%DEHT from toy turtle on steel
Time_Turtle_DEHT=Data_2(51:56,1); %h
Data_Turtle_DEHT=Data_2(51:56,2); %ng

%ATBC from toy turtle on steel
Time_Turtle_ATBC=Data_2(51:56,1); %h
Data_Turtle_ATBC=Data_2(51:56,3); %ng

%DEHT from toy giraffe on steel
Time_Giraffe_DEHT=Data_2(59:65,1); %h
Data_Giraffe_DEHT=Data_2(59:65,2); %ng

%ATBC from toy giraffe on steel
Time_Giraffe_ATBC=Data_2(59:65,1); %h
Data_Giraffe_ATBC=Data_2(59:65,3); %ng

%DEHT from small dinosaur toy on steel
Time_Smalldinosaur_DEHT=Data_2(68:73,1); %h
Data_Smalldinosaur_DEHT=Data_2(68:73,2); %ng

%DEHT from large dinosaur toy on steel
Time_Largedinosaur_DEHT=Data_2(76:81,1); %h
Data_Largedinosaur_DEHT=Data_2(76:81,2); %ng

%%%%%Figures which will be outputted%%%%%%%

figure(1) %DEHP on different surfaces
subplot(2,4,1)
errorbar(Time_DEHP_Aluminum,Data_DEHP_Aluminum,Error_DEHP_Aluminum,'o')
hold on
plot(t_hour_1(:,1),aa_DEHP_ng_output(:,1))
title('DEHP aluminum')

subplot(2,4,2)
errorbar(Time_DEHP_Smoothglass,Data_DEHP_Smoothglass,Error_DEHP_Smoothglass,'o')
title('DEHP Smooth glass')
hold on
plot(t_hour_1(:,2),aa_DEHP_ng_output(:,2))

subplot(2,4,3)
errorbar(Time_DEHP_Roughglass,Data_DEHP_Roughglass,Error_DEHP_Roughglass,'o')
title('DEHP Rough glass')
hold on
plot(t_hour_1(:,3),aa_DEHP_ng_output(:,3))

subplot(2,4,4)
errorbar(Time_DEHP_Stainlesssteel_1,Data_DEHP_Stainlesssteel_1,Error_DEHP_Stainlesssteel_1,'o')
title('DEHP Stainless steel 1')
hold on
plot(t_hour_1(:,4),aa_DEHP_ng_output(:,4))

subplot(2,4,5)
errorbar(Time_DEHP_Acrylicplastic,Data_DEHP_Acrylicplastic,Error_DEHP_Acrylicplastic,'o')
title(' DEHP Acrylic plastic')
hold on
plot(t_hour_1(:,5),aa_DEHP_ng_output(:,5))

subplot(2,4,6)
errorbar(Time_DEHP_steel,Data_DEHP_steel,Error_DEHP_steel,'o')
title('DEHP steel')
hold on
plot(t_hour_1(:,6),aa_DEHP_ng_output(:,6))

subplot(2,4,7)
errorbar(Time_DEHP_Stainlesssteel_2,Data_DEHP_Stainlesssteel_2,Error_DEHP_Stainlesssteel_2,'o')
title('DEHP Stainless steel 2')
hold on
plot(t_hour_1(:,7),aa_DEHP_ng_output(:,7))

fig=figure(1);
han=axes(fig,'visible','off'); 
han.Title.Visible='on';
han.XLabel.Visible='on';
han.YLabel.Visible='on';
ylabel(han,'Mass on the surface (ng)');
xlabel(han,'Time (h)');
sgtitle('DEHP on different surfaces');

figure(2) %Various phthalates on steel and stainless steel
subplot(2,5,1)
errorbar(Time_Backpack_R0b_DEHT,Data_Backpack_R0b_DEHT,Error_Backpack_R0b_DEHT,'o')
title('Backpack R0b DEHT (SS)')
hold on
plot(t_hour_1(:,8),aa_DEHT_ng_output(:,8))

subplot(2,5,2)
errorbar(Time_Backpack_STb_DIDP,Data_Backpack_STb_DIDP,Error_Backpack_STb_DIDP,'o')
title('Backpack STb DIDP (SS)')
hold on
plot(t_hour_1(:,9),aa_DIDP_ng_output(:,9))

subplot(2,5,3)
errorbar(Time_Backpack_Tab_DEHP,Data_Backpack_Tab_DEHP,Error_Backpack_Tab_DEHP,'o')
title('Backpack Tab DEHP (SS)')
hold on
plot(t_hour_1(:,10),aa_DEHP_ng_output(:,10))

subplot(2,5,4)
errorbar(Time_Backpack_Tab_DEHA,Data_Backpack_Tab_DEHA,Error_Backpack_Tab_DEHA,'o')
title('Backpack Tab DEHA (SS)')
hold on
plot(t_hour_1(:,10),aa_DEHA_ng_output(:,10))

subplot(2,5,5)
errorbar(Time_Backpack_TMb_DEHT,Data_Backpack_TMb_DEHT,Error_Backpack_TMb_DEHT,'o')
title('Backpack TMb DEHT (SS)')
hold on
plot(t_hour_1(:,11),aa_DEHT_ng_output(:,11))

subplot(2,5,6)
errorbar(Time_Backpack_TMb_DINP,Data_Backpack_TMb_DINP,Error_Backpack_TMb_DINP,'o')
title('Backpack TMb DINP (SS)')
hold on
plot(t_hour_1(:,11),aa_DINP_ng_output(:,11))

subplot(2,5,7)
errorbar(Time_Backpack_WAb1_DEHT,Data_Backpack_WAb1_DEHT,Error_Backpack_WAb1_DEHT,'o')
title('Backpack WAb1 DEHT (SS)')
hold on
plot(t_hour_1(:,12),aa_DEHT_ng_output(:,12))

subplot(2,5,8)
errorbar(Time_Backpack_WAb1_DEHA,Data_Backpack_WAb1_DEHA,Error_Backpack_WAb1_DEHA,'o')
title('Backpack WAb1 DEHA (SS)')
hold on
plot(t_hour_1(:,12),aa_DEHA_ng_output(:,12))

subplot(2,5,9)
errorbar(Time_Backpack_WAb2_DEHP,Data_Backpack_WAb2_DEHP,Error_Backpack_WAb2_DEHP,'o')
title('Backpack WAb2 DEHP (steel)')
hold on
plot(t_hour_1(:,13),aa_DEHP_ng_output(:,13))

subplot(2,5,10)
errorbar(Time_Backpack_WAb2_DEHT,Data_Backpack_WAb2_DEHT,Error_Backpack_WAb2_DEHT,'o')
title('Backpack WAb2 DEHT (steel)')
hold on
plot(t_hour_1(:,13),aa_DEHT_ng_output(:,13))

fig=figure(2);
han=axes(fig,'visible','off'); 
han.Title.Visible='on';
han.XLabel.Visible='on';
han.YLabel.Visible='on';
ylabel(han,'Mass on the surface (ng)');
xlabel(han,'Time (h)');
sgtitle('Phthalates from various objects on stainless steel (SS) or steel');

figure(3) %Various phthalates on steel
subplot(2,5,1)
errorbar(Time_Backpack_WAb2_DIDP,Data_Backpack_WAb2_DIDP,Error_Backpack_WAb2_DIDP,'o')
title('Backpack WAb2 DIDP')
hold on
plot(t_hour_1(:,13),aa_DIDP_ng_output(:,13))

subplot(2,5,2)
errorbar(Time_Backpack_WAb2_DINP,Data_Backpack_WAb2_DINP,Error_Backpack_WAb2_DINP,'o')
title('Backpack WAb2 DINP')
hold on
plot(t_hour_1(:,13),aa_DINP_ng_output(:,13))

subplot(2,5,3)
errorbar(Time_Backpack_WAb2_DINCH,Data_Backpack_WAb2_DINCH,Error_Backpack_WAb2_DINCH,'o')
title('Backpack WAb2 DINCH')
hold on
plot(t_hour_1(:,13),aa_DINCH_ng_output(:,13))

subplot(2,5,4)
scatter(Time_Turtle_DEHT,Data_Turtle_DEHT,'o')
title('Toy turtle DEHT')
hold on
plot(t_hour_1(:,14),aa_DEHT_ng_output(:,14))

subplot(2,5,5)
scatter(Time_Turtle_ATBC,Data_Turtle_ATBC,'o')
title('Toy turtle ATBC')
hold on
plot(t_hour_1(:,14),aa_ATBC_ng_output(:,14))

subplot(2,5,6)
scatter(Time_Giraffe_DEHT,Data_Giraffe_DEHT,'o')
title('Toy giraffe DEHT')
hold on
plot(t_hour_1(:,15),aa_DEHT_ng_output(:,15))

subplot(2,5,7)
scatter(Time_Giraffe_ATBC,Data_Giraffe_ATBC,'o')
title('Toy giraffe ATBC')
hold on
plot(t_hour_1(:,15),aa_ATBC_ng_output(:,15))

subplot(2,5,8)
scatter(Time_Smalldinosaur_DEHT,Data_Smalldinosaur_DEHT,'o')
title('Toy small dinosaur DEHT')
hold on
plot(t_hour_1(:,16),aa_DEHT_ng_output(:,16))

subplot(2,5,9)
scatter(Time_Largedinosaur_DEHT,Data_Largedinosaur_DEHT,'o')
title('Toy large dinosaur DEHT')
hold on
plot(t_hour_1(:,17),aa_DEHT_ng_output(:,17))

fig=figure(3);
han=axes(fig,'visible','off'); 
han.Title.Visible='on';
han.XLabel.Visible='on';
han.YLabel.Visible='on';
ylabel(han,'Mass on the surface (ng)');
xlabel(han,'Time (h)');
sgtitle('Phthalates from various objects on steel');
  